#!/usr/bin/env python3
# F17_v6 — Two-Source Interference (No-Signalling + Ping-Pong Overlap + Co-moving Envelope)
# Control: present-act, boolean/ordinal; one commit per tick; NO RNG.
# - COH: opposite sweeps; overlap commits scan deterministically across the actual overlap arc
#        around the instantaneous midpoint (ping-pong reflection at edges).
# - REF: same-direction; on overlap and union use A/B parity alternation (no midpoint) to keep baseline flat.
# Diagnostics-only: envelopes computed via circular smoothing of histograms
# - COH in a co-moving frame (rotate by -midpoint); REF in the fixed frame.
# Acceptance gates (example): V_coh>=0.25, V_ref<=0.07, r2_coh>=0.80, flat<=0.40, totals_equal=True.

import argparse, csv, json, math, os, sys
from typing import List

def modS(x: int, S: int) -> int:
    x %= S
    return x if x >= 0 else x + S

def arc_window(center: int, half: int, S: int) -> List[int]:
    return [modS(center + k, S) for k in range(-half, half+1)]

def circ_mid(a: int, b: int, S: int) -> int:
    # vector-average midpoint on the ring; deterministic tie-break for opposite points
    ang_a = 2.0*math.pi*(a/S); ang_b = 2.0*math.pi*(b/S)
    cx = math.cos(ang_a) + math.cos(ang_b)
    cy = math.sin(ang_a) + math.sin(ang_b)
    if abs(cx) < 1e-12 and abs(cy) < 1e-12:
        return min(a, b)
    ang = math.atan2(cy, cx)
    if ang < 0: ang += 2.0*math.pi
    return int(round((ang/(2.0*math.pi))*S)) % S

def circ_conv(xs: List[float], kernel: List[float]) -> List[float]:
    S=len(xs); L=len(kernel); h=L//2
    out=[0.0]*S; ksum=sum(kernel) if sum(kernel)!=0 else 1.0
    for j in range(S):
        acc=0.0
        for i in range(L):
            jj=(j + i - h) % S
            acc += kernel[i]*xs[jj]
        out[j]=acc/ksum
    return out

def vis(xs: List[float]) -> float:
    mx=max(xs); mn=min(xs)
    return (mx-mn)/(mx+mn) if (mx+mn)>0 else 0.0

def flat_rel_rmse(xs: List[float]) -> float:
    mu=sum(xs)/len(xs)
    if mu==0: return float("inf")
    return (sum((x-mu)*(x-mu) for x in xs)/len(xs))**0.5 / mu

def r2_first_harmonic(xs: List[float]) -> float:
    S=len(xs); mu=sum(xs)/S
    cosv=[math.cos(2.0*math.pi*k/S) for k in range(S)]
    sinv=[math.sin(2.0*math.pi*k/S) for k in range(S)]
    c=sum((xs[i]-mu)*cosv[i] for i in range(S))
    s=sum((xs[i]-mu)*sinv[i] for i in range(S))
    rec=[mu + (c*cosv[i] + s*sinv[i])/(S/2.0) for i in range(S)]
    ss_tot=sum((xs[i]-mu)*(xs[i]-mu) for i in range(S))
    ss_res=sum((xs[i]-rec[i])*(xs[i]-rec[i]) for i in range(S))
    return 1.0 - (ss_res/ss_tot if ss_tot>0 else 0.0)

def run_panel(panel: str, P: dict) -> dict:
    S, H = int(P["S"]), int(P["H"])
    sA0, sB0 = int(P["sA0"]), int(P["sB0"])
    wA, wB = int(P["wA"]), int(P["wB"])
    stepA, stepB_coh, stepB_ref = int(P["stepA"]), int(P["stepB_coh"]), int(P["stepB_ref"])

    raw = [0]*S        # fixed-frame histogram
    rot = [0]*S        # co-moving histogram (COH only)

    scan_offset = 0    # ping-pong state for COH overlap
    scan_dir = +1

    for t in range(H):
        if panel == "COH":
            sA_t = modS(sA0 - stepA*t, S)      # CW
            sB_t = modS(sB0 + stepB_coh*t, S)  # CCW
        else:
            sA_t = modS(sA0 - stepA*t, S)      # CW
            sB_t = modS(sB0 - stepB_ref*t, S)  # CW

        Awin = set(arc_window(sA_t, wA, S))
        Bwin = set(arc_window(sB_t, wB, S))
        overlap = Awin & Bwin

        if overlap:
            if panel == "COH":
                mid = circ_mid(sA_t, sB_t, S)
                # find symmetric radius r so mid+[-r..r] subset of overlap
                r = 0
                while (r+1) <= min(wA, wB):
                    ok = True
                    for k in range(-(r+1), (r+1)+1):
                        if modS(mid + k, S) not in overlap:
                            ok = False; break
                    if not ok: break
                    r += 1
                # choose index around mid using scan_offset with clamping
                if scan_offset > r: scan_offset = r
                if scan_offset < -r: scan_offset = -r
                j = modS(mid + scan_offset, S)
                if j not in overlap:
                    # fallback: nearest in overlap to mid (min circular distance; tie -> smallest index)
                    best = None; bestd = None
                    for s in overlap:
                        d = min((s - mid) % S, (mid - s) % S)
                        if best is None or d < bestd or (d == bestd and s < best):
                            best, bestd = s, d
                    j = best
                raw[j] += 1
                # advance ping-pong
                scan_offset += scan_dir
                if scan_offset > r or scan_offset < -r:
                    scan_dir *= -1
                    scan_offset += 2*scan_dir  # reflect
            else:
                # REF: use union alternation even on overlap
                j = sA_t if (t % 2 == 0) else sB_t
                raw[j] += 1
        else:
            # union case for both panels
            j = sA_t if (t % 2 == 0) else sB_t
            raw[j] += 1

        if panel == "COH":
            mid = circ_mid(sA_t, sB_t, S)
            j_rot = modS(j - mid, S)
            rot[j_rot] += 1

    # Triangular kernel length 61 (diagnostics)
    kernel = [i+1 for i in range(30)] + [31] + [30-i for i in range(30)]

    if panel == "COH":
        env = circ_conv([float(x) for x in rot], kernel)          # co-moving envelope
        fixed_env = circ_conv([float(x) for x in raw], kernel)    # transparency
        return {
            "total": sum(raw),
            "env": env,
            "fixed_env": fixed_env,
            "V": vis(env),
            "flat": flat_rel_rmse(env),
            "r2": r2_first_harmonic(env),
            "raw_counts": raw
        }
    else:
        env = circ_conv([float(x) for x in raw], kernel)          # fixed-frame envelope
        return {
            "total": sum(raw),
            "env": env,
            "V": vis(env),
            "flat": flat_rel_rmse(env),
            "r2": r2_first_harmonic(env),
            "raw_counts": raw
        }

def main():
    ap = argparse.ArgumentParser()
    ap.add_argument("--manifest", required=True)
    ap.add_argument("--outdir", required=True)
    args = ap.parse_args()

    # Load manifest and prepare outdir
    with open(args.manifest, "r", encoding="utf-8") as f:
        M = json.load(f)
    os.makedirs(args.outdir, exist_ok=True)
    for sub in ["config", "outputs/metrics", "outputs/audits", "outputs/run_info", "logs"]:
        os.makedirs(os.path.join(args.outdir, sub), exist_ok=True)

    # snapshot manifest
    man_copy = os.path.join(args.outdir, "config", "manifest_f17_v6.json")
    with open(man_copy, "w", encoding="utf-8") as f:
        json.dump(M, f, indent=2, sort_keys=True)

    S = int(M["sectors"]["S"]); H = int(M["H"])
    P = {
        "S": S, "H": H,
        "sA0": int(M["sources"]["sA0"]), "sB0": int(M["sources"]["sB0"]),
        "wA": int(M["sources"]["wA_half"]), "wB": int(M["sources"]["wB_half"]),
        "stepA": int(M["sources"]["stepA"]),
        "stepB_coh": int(M["sources"]["stepB_coh"]),
        "stepB_ref": int(M["sources"]["stepB_ref"])
    }

    coh = run_panel("COH", P)
    ref = run_panel("REF", P)

    # Metrics CSVs
    with open(os.path.join(args.outdir, "outputs/metrics", "f17_coh_sector_counts.csv"), "w", newline="", encoding="utf-8") as f:
        w = csv.writer(f); w.writerow(["sector","count"])
        for j,c in enumerate(coh["raw_counts"]): w.writerow([j, c])
    with open(os.path.join(args.outdir, "outputs/metrics", "f17_ref_sector_counts.csv"), "w", newline="", encoding="utf-8") as f:
        w = csv.writer(f); w.writerow(["sector","count"])
        for j,c in enumerate(ref["raw_counts"]): w.writerow([j, c])

    # Acceptance & no-signalling
    totals_equal = (coh["total"] == ref["total"] == H)
    V_min = float(M["acceptance"]["V_min_coh"])
    V_max = float(M["acceptance"]["V_max_ref"])
    r2_min = float(M["acceptance"]["r2_min_coh"])
    flat_max = float(M["acceptance"]["flat_rel_rmse_max"])

    passed = bool(
        totals_equal and
        (coh["V"] >= V_min) and (ref["V"] <= V_max) and
        (coh["r2"] >= r2_min) and
        (coh["flat"] <= flat_max) and (ref["flat"] <= flat_max)
    )

    audit = {
        "sim": "F17_two_source_v6",
        "S": S, "H": H,
        "panels": {
            "COH": {"total": coh["total"], "V": coh["V"], "flat_rel_rmse": coh["flat"], "r2_first_harm": coh["r2"]},
            "REF": {"total": ref["total"], "V": ref["V"], "flat_rel_rmse": ref["flat"], "r2_first_harm": ref["r2"]}
        },
        "no_signalling": {"totals_equal": totals_equal},
        "accept": {
            "V_min_coh": V_min, "V_max_ref": V_max,
            "r2_min_coh": r2_min, "flat_rel_rmse_max": flat_max
        },
        "pass": passed
    }
    with open(os.path.join(args.outdir, "outputs", "audits", "f17_audit.json"), "w", encoding="utf-8") as f:
        json.dump(audit, f, indent=2, sort_keys=True)

    result_line = ("F17_v6 PASS={p} V_coh={vc:.3f} V_ref={vr:.3f} "
                   "r2_coh={r2c:.3f} flat_coh={fc:.3f} flat_ref={fr:.3f} totals_equal={te}"
                   .format(p=passed, vc=coh["V"], vr=ref["V"], r2c=coh["r2"],
                           fc=coh["flat"], fr=ref["flat"], te=totals_equal))
    with open(os.path.join(args.outdir, "outputs", "run_info", "result_line.txt"), "w", encoding="utf-8") as f:
        f.write(result_line)
    print(result_line)

if __name__ == "__main__":
    main()
